/**
 * The 2 dimensional grid of rules with some utilities.
 * 
 * @author TineL
 * @version 1.0, 2007-05-30
 * @since 1.0
 */
public class Rules extends Abstract2DGrid {

  /** 2D array of rules. */
  private Rule[][] rules;

  /**
   * Creates <code>Rules</code> with the specified size. All rules are set to
   * 0 code.
   * 
   * @param width the grid width.
   * @param height the grid height.
   * @since 1.0
   */
  public Rules(int width, int height) {
    super(width, height);
    this.rules=new Rule[height][width];
  }

  /**
   * Creates <code>Rules</code> from the given rules.
   * 
   * @param rule the rule (<code>null</code> not permitted).
   * @since 1.0
   */
  public Rules(Rules rule) {
    this(rule.getWidth(), rule.getHeight());
    for (int y=0; y<getHeight(); y++) {
      for (int x=0; x<getHeight(); x++) {
        rules[y][x]=rule.get(x, y);
      }
    }
  }

  /**
   * Generates a grid of random rules.
   * 
   * @param width the grid width.
   * @param height the grid height.
   * @return a grid of random rules (<code>null</code> not possible).
   * @since 1.0
   */
  public static Rules generateRandomRules(int width, int height) {
    Rules rules=new Rules(width, height);
    for (int y=0; y<height; y++) {
      for (int x=0; x<width; x++) {
        rules.set(x, y, Rule.generateRandomRule());
      }
    }
    return rules;
  }

  /**
   * Generates a grid of specific rules defined by the ruleCode. All generated
   * rules are the same.
   * 
   * @param width the grid width.
   * @param height the grid height.
   * @param ruleCode the rule code for all rules.
   * @return a grid of rules (<code>null</code> not possible).
   * @since 1.0
   */
  public static Rules generateRules(int width, int height, int ruleCode) {
    Rules rules=new Rules(width, height);
    for (int y=0; y<height; y++) {
      for (int x=0; x<width; x++) {
        rules.set(x, y, new Rule(ruleCode));
      }
    }
    return rules;
  }

  /**
   * @param x the rule X coordinate.
   * @param y the rule Y coordinate.
   * @return the rule (<code>null</code> not possible).
   * @since 1.0
   */
  public Rule get(int x, int y) {
    return rules[y][x];
  }

  /**
   * Replaces the old rule with the given rule at the specified location.
   * 
   * @param x the rule X coordinate.
   * @param y the rule Y coordinate.
   * @param rule the new rule (<code>null</code> not permitted).
   * @since 1.0
   */
  public void set(int x, int y, Rule rule) {
    rules[y][x]=rule;
  }

  /**
   * @param x the rule X coordinate.
   * @param y the rule Y coordinate.
   * @return the rule above (neighbour) (<code>null</code> not possible).
   * @since 1.0
   */
  public Rule getAbove(int x, int y) {
    return get(x, getAboveIndex(y));
  }

  /**
   * @param x the rule X coordinate.
   * @param y the rule Y coordinate.
   * @return the rule on the right (neighbour) (<code>null</code> not
   *         possible).
   * @since 1.0
   */
  public Rule getRight(int x, int y) {
    return get(getRightIndex(x), y);
  }

  /**
   * @param x the rule X coordinate.
   * @param y the rule Y coordinate.
   * @return the rule below (neighbour) (<code>null</code> not possible).
   * @since 1.0
   */
  public Rule getBelow(int x, int y) {
    return get(x, getBelowIndex(y));
  }

  /**
   * @param x the rule X coordinate.
   * @param y the rule Y coordinate.
   * @return the rule on the left (neighbour) (<code>null</code> not
   *         possible).
   * @since 1.0
   */
  public Rule getLeft(int x, int y) {
    return get(getLeftIndex(x), y);
  }

  /*
   * @since 1.0
   */
  public String toString() {
    String out="Rules:";
    for (int y=0; y<getHeight(); y++) {
      out+="\n[";
      for (int x=0; x<getWidth()-1; x++) {
        out+=get(x, y).toHexString()+" ";
      }
      out+=get(getWidth()-1, y).toHexString()+"]";
    }
    return out;
  }
}
