/**
 * The 32-bit rule represented with the code. Code is implemented with one
 * integer. Bits are represented as binary integer number. Bit index 0 is the
 * least significant bit, index 30 is the most significant bit. Handling with
 * rule is done by the boolean values: <code>true</code> represents 1 and
 * <code>false</code> represents 0.
 * 
 * @author TineL
 * @version 1.0, 2007-05-30
 * @since 1.0
 */
public class Rule {

  /** Rule code. */
  private int code=0;

  /**
   * Creates <code>Rule</code> with zero code.
   * 
   * @since 1.0
   */
  public Rule() {}

  /**
   * Creates <code>Rule</code> with the given code.
   * 
   * @param ruleCode the rule code.
   * @since 1.0
   */
  public Rule(int ruleCode) {
    this.code=ruleCode;
  }

  /**
   * Creates <code>Rule</code> from the specified rule.
   * 
   * @param rule the rule (<code>null</code> not permitted).
   * @since 1.0
   */
  public Rule(Rule rule) {
    this(rule.getCode());
  }

  /**
   * Generates a random rule.
   * 
   * @return the random rule (<code>null</code> not possible).
   * @since 1.0
   */
  public static Rule generateRandomRule() {
    // Create negative & positive random integers
    return new Rule((int)(Math.random()*Integer.MAX_VALUE)*2);
  }

  /**
   * @return the rule code.
   * @since 1.0
   */
  private int getCode() {
    return code;
  }

  /**
   * @param ruleCode the rule code.
   * @since 1.0
   */
  private void setCode(int ruleCode) {
    this.code=ruleCode;
  }

  /**
   * @param index the integer index of the bit.
   * @return a 2<sup>index</sup> number.
   * @throws IndexOutOfBoundsException if index is not in [0, 31].
   * @since 1.0
   */
  private int bitIndex(int index) {
    if (index<0|index>31)
      throw new IndexOutOfBoundsException("Index can only be [0, 31].");
    return 1<<index; // 2^index
  }

  /**
   * @param index the index of the bit (must be [0, 31]).
   * @return <code>true</code> if bit is 1, <code>false</code> if bit is 0.
   * @since 1.0
   */
  public boolean get(int index) {
    int bit=bitIndex(index);
    return (bit&getCode())==bit;
  }

  /**
   * Set the bit at the specified index to <code>false</code>.
   * 
   * @param index the index of the bit (must be [0, 31]).
   * @since 1.0
   */
  public void clear(int index) {
    int bit=bitIndex(index);
    setCode(~bit&getCode());
  }

  /**
   * Set the bit at the specified index to <code>true</code>.
   * 
   * @param index the index of the bit (must be [0, 31]).
   * @since 1.0
   */
  public void set(int index) {
    int bit=bitIndex(index);
    setCode(bit|getCode());
  }

  /**
   * Set the bit at the specified index to the specified value.
   * 
   * @param index the index of the bit (must be [0, 31]).
   * @param value
   * @since 1.0
   */
  public void set(int index, boolean value) {
    if (value) {
      set(index);
    } else {
      clear(index);
    }
  }

  /**
   * Flips the bit at the specified index to represent a negated value.
   * 
   * @param index the index of the bit (must be [0, 31]).
   * @since 1.0
   */
  public void flip(int index) {
    set(index, !get(index));
  }

  /**
   * Mutates (damages) the rule with the specified probability. This is done by
   * flipping bits. Note that original rule remains untouched.
   * 
   * @param probability the probability to flip the bit.
   * @return the mutated rule (<code>null</code> not possible).
   * @since 1.0
   */
  public Rule mutate(double probability) {
    Rule rule=new Rule(this);
    for (int i=0; i<32; i++) {
      if (Math.random()<probability) rule.flip(i);
    }
    return rule;
  }

  /**
   * Traverses the cell's grid with the rule by specifying neighbour cell states
   * and getting the cell's new state. States are represented as bits in the
   * rule table.<br>
   * 
   * @param fifth the state of the cell (most significant bit).
   * @param forth the state of the cell.
   * @param third the state of the cell.
   * @param second the state of the cell.
   * @param first the state of the cell (least significant bit).
   * @return the state of the new cell.
   * @since 1.0
   */
  public boolean traverse(boolean fifth, boolean forth, boolean third,
    boolean second, boolean first) {
    // Convert bits to code
    int fi=fifth?16:0;
    int fo=forth?8:0;
    int th=third?4:0;
    int nd=second?2:0;
    int st=first?1:0;
    return get(fi|fo|th|nd|st);
  }

  /**
   * @return a string representation of the rule code in hexadecimal notation (<code>null</code>
   *         not possible).
   * @since 1.0
   */
  public String toHexString() {
    return "0x"+Integer.toHexString(getCode());
  }

  /*
   * @since 1.0
   */
  public String toString() {
    String output="Rule=[";
    for (int i=31; i>=0; i--) {
      output+=get(i)?"1":"0";
    }
    return output+"]="+toHexString();
  }
}
