/**
 * The 2 dimensional grid for storing cell fitnesses with some utilities.
 * 
 * @author TineL
 * @version 1.0, 2007-05-30
 * @since 1.0
 */
public class Fitnesses extends Abstract2DGrid {

  /** 2D array of fitnesses. */
  private int[][] fitnesses;

  /**
   * Creates <code>Fitnesses</code> with the specified size. All fitnesses are
   * set to 0.
   * 
   * @param width the grid width.
   * @param height the grid height.
   * @since 1.0
   */
  public Fitnesses(int width, int height) {
    super(width, height);
    this.fitnesses=new int[height][width];
  }

  /**
   * @param x the fitness X coordinate.
   * @param y the fitness Y coordinate.
   * @return the fitness.
   * @since 1.0
   */
  public int get(int x, int y) {
    return fitnesses[y][x];
  }

  /**
   * Increments the fitness at the specified location for one.
   * 
   * @param x the fitness X coordinate.
   * @param y the fitness Y coordinate.
   * @since 1.0
   */
  public void inc(int x, int y) {
    fitnesses[y][x]++;
  }

  /**
   * @return the sum of all fitnesses in the grid.
   * @since 1.0
   */
  public int sum() {
    int sum=0;
    for (int y=0; y<getHeight(); y++) {
      for (int x=0; x<getWidth(); x++) {
        sum+=get(x, y);
      }
    }
    return sum;
  }

  /**
   * @param x the fitness X coordinate.
   * @param y the fitness Y coordinate.
   * @return the above fitness.
   * @since 1.0
   */
  public int getAbove(int x, int y) {
    int uy=getAboveIndex(y);
    if (uy==-1) return 0;
    return get(x, uy);
  }

  /**
   * @param x the fitness X coordinate.
   * @param y the fitness Y coordinate.
   * @return the right fitness.
   * @since 1.0
   */
  public int getRight(int x, int y) {
    int rx=getRightIndex(x);
    if (rx==-1) return 0;
    return get(rx, y);
  }

  /**
   * @param x the fitness X coordinate.
   * @param y the fitness Y coordinate.
   * @return the below fitness.
   * @since 1.0
   */
  public int getBelow(int x, int y) {
    int ly=getBelowIndex(y);
    if (ly==-1) return 0;
    return get(x, ly);
  }

  /**
   * @param x the fitness X coordinate.
   * @param y the fitness Y coordinate.
   * @return the left fitness.
   * @since 1.0
   */
  public int getLeft(int x, int y) {
    int lx=getLeftIndex(x);
    if (lx==-1) return 0;
    return get(lx, y);
  }

  /**
   * @param x the fitness X coordinate.
   * @param y the fitness Y coordinate.
   * @return <code>true</code> if above fitness is bigger.
   * @since 1.0
   */
  public boolean isAboveBigger(int x, int y) {
    return getAbove(x, y)>get(x, y);
  }

  /**
   * @param x the fitness X coordinate.
   * @param y the fitness Y coordinate.
   * @return <code>true</code> if right fitness is bigger.
   * @since 1.0
   */
  public boolean isRightBigger(int x, int y) {
    return getRight(x, y)>get(x, y);
  }

  /**
   * @param x the fitness X coordinate.
   * @param y the fitness Y coordinate.
   * @return <code>true</code> if below fitness is bigger.
   * @since 1.0
   */
  public boolean isBelowBigger(int x, int y) {
    return getBelow(x, y)>get(x, y);
  }

  /**
   * @param x the fitness X coordinate.
   * @param y the fitness Y coordinate.
   * @return <code>true</code> if left fitness is bigger.
   * @since 1.0
   */
  public boolean isLeftBigger(int x, int y) {
    return getLeft(x, y)>get(x, y);
  }

  /**
   * Computes how many fitnesses around the specified one has bigger values.
   * Neighbour fitnesses are one above, one right, one below, and one left.
   * 
   * @param x the fitness X coordinate.
   * @param y the fitness Y coordinate.
   * @return the number of bigger fitnesses around (only 0, 1, 2, 3 or 4).
   * @since 1.0
   */
  public int biggerNeighbours(int x, int y) {
    int num=0;
    if (isAboveBigger(x, y)) num++;
    if (isRightBigger(x, y)) num++;
    if (isBelowBigger(x, y)) num++;
    if (isLeftBigger(x, y)) num++;
    return num;
  }

  /*
   * @since 1.0
   */
  public String toString() {
    String out="Fitnesses:";
    for (int y=0; y<getHeight(); y++) {
      out+="\n[";
      for (int x=0; x<getWidth()-1; x++) {
        out+=get(x, y)+" ";
      }
      out+=get(getWidth()-1, y)+"]";
    }
    return out;
  }
}
