/**
 * This is a cellular programming algorithm (CPA) for rectangle-boundary task
 * developed from the Moshe Sippers CPA algorithm. This CPA has 2 states and is
 * non-uniform. Initial rules are random, cellular automata (CA) are also
 * random. Rules are 32 bits long. CA is 2 dimensional, each cell has 5
 * neighbours (itself, above, right, below, and left - von Neumann type).
 * Spatially periodic boundary conditions are not applied.
 * 
 * @author TineL
 * @version 1.0, 2007-05-30
 * @since 1.0
 */
public class CPA {

  /** Number of cells horizontally. */
  private final static int GRID_WIDTH=8;
  /** Number of cells vertically. */
  private final static int GRID_HEIGHT=8;

  /** Number od execution steps. */
  private final static int STEP_NUM=12;
  /** Number of start configurations. */
  private final static int START_CONFIG_NUM=200;
  /** Number of generations. */
  private final static int GENERATION_NUM=2000;

  /** Rule mutation probability. */
  private static double MUTATION_PROBABILITY=0.001;

  /** Are the rectangle corner cells in state 1? */
   private final static boolean ONES_IN_CORNERS=true;
//  private final static boolean ONES_IN_CORNERS=false;

  /**
   * Start method.
   * 
   * @param args the command line arguments (not used).
   * @since 1.0
   */
  public static void main(String[] args) {
    long time=System.currentTimeMillis();

    CPA cpa=new CPA();
    cpa.run();

    long duration=System.currentTimeMillis()-time;
    System.out.println("Run for "+((double)duration/1000)+" s.");
  }

  /**
   * Runs the CPA.
   * 
   * @since 1.0
   */
  public void run() {
    // Generate rule set
    System.out.println("Generating rules...");
    Rules rules=Rules.generateRandomRules(GRID_WIDTH, GRID_HEIGHT);
    System.out.println("  "+rules);

    double bestFitnessSuccess=0;
    Fitnesses bestFitnesses=null;
    Rules bestRules=null;

    // Generations
    for (int g=0; g<GENERATION_NUM; g++) {
      System.out.println("Starting new generation #"+(g+1)+"...");

      Fitnesses fitnesses=new Fitnesses(GRID_WIDTH, GRID_HEIGHT);

      // ////////////////////////////////////////
      // Rules evaluation
      // ////////////////////////////////////////

      for (int c=0; c<START_CONFIG_NUM; c++) {
        // Define the rectangle boundaries
        int leftEdge=1+(int)(Math.random()*(GRID_WIDTH/2-1));
        int rightEdge=GRID_WIDTH/2+(int)(Math.random()*(GRID_WIDTH/2-1));
        int topEdge=1+(int)(Math.random()*(GRID_HEIGHT/2-1));
        int bottomEdge=GRID_HEIGHT/2+(int)(Math.random()*(GRID_HEIGHT/2-1));

        // Generate initial configuration
        double onesRatio=Math.random();
        CA ca=CA.generateRandomCA(GRID_WIDTH, GRID_HEIGHT, topEdge, rightEdge,
          leftEdge, bottomEdge, onesRatio);
        if (ONES_IN_CORNERS) {
          ca.set(leftEdge, topEdge, true);
          ca.set(leftEdge, bottomEdge, true);
          ca.set(rightEdge, topEdge, true);
          ca.set(rightEdge, bottomEdge, true);
        }

        // Execution
        for (int m=0; m<STEP_NUM; m++) {
          CA caNext=new CA(GRID_WIDTH, GRID_HEIGHT);
          for (int y=0; y<GRID_HEIGHT; y++) {
            for (int x=0; x<GRID_WIDTH; x++) {
              caNext.set(x, y, rules.get(x, y).traverse(ca.get(x, y),
                ca.getAbove(x, y), ca.getRight(x, y), ca.getBelow(x, y),
                ca.getLeft(x, y)));
            }
          }
          ca=caNext;
        }

        // Fitness
        for (int y=0; y<GRID_HEIGHT; y++) {
          for (int x=0; x<GRID_WIDTH; x++) {
            if (y>=topEdge&&y<=bottomEdge&&x>=leftEdge&&x<=rightEdge) {
              // Inside rectangle
              if (ca.get(x, y)) {
                // Cell state is 1
                fitnesses.inc(x, y);
              }
            } else {
              // Outside rectangle
              if (!ca.get(x, y)) {
                // Cell state is 0
                fitnesses.inc(x, y);
              }
            }
          }
        }
      }

      // ////////////////////////////////////////
      // Fitness analyze
      // ////////////////////////////////////////

      int fitnessSum=fitnesses.sum();
      double fitnessSuccess=(double)fitnessSum
        /(GRID_WIDTH*GRID_HEIGHT*START_CONFIG_NUM)*100;

      // Print out analysis if fitness is best in this generation
      if (fitnessSuccess>bestFitnessSuccess) {
        System.out.println("  "+fitnesses);
        System.out.println("  fitnessSuccess="+fitnessSuccess+"%");
        System.out.println("  "+rules);
        bestFitnesses=fitnesses;
        bestFitnessSuccess=fitnessSuccess;
        bestRules=rules;
      }
      if (bestFitnessSuccess==100.0) break; // End

      // ////////////////////////////////////////
      // Rules evolution
      // ////////////////////////////////////////

      Rules newRules=new Rules(rules);

      for (int y=0; y<GRID_HEIGHT; y++) {
        for (int x=0; x<GRID_WIDTH; x++) {
          // Get better neighbour rules for a cell
          Rule[] betterRules=findBetterRules(fitnesses, rules, x, y);

          if (betterRules==null) {
            // No rule is better

            // Rule is left unchanged

          } else if (betterRules.length==1) {
            // Above or right or below or left rule is better

            // Better rule is used
            newRules.set(x, y, betterRules[0]);

            // Mutate
            newRules.set(x, y, newRules.get(x, y).mutate(MUTATION_PROBABILITY));

          } else if (betterRules.length>=2) {
            // Two or more rules are better

            // Randomly select which better rules will be crossed over
            int ruleIndex1=(int)(Math.random()*betterRules.length);
            int ruleIndex2=(int)(Math.random()*(betterRules.length-1));
            if (ruleIndex2>=ruleIndex1) ruleIndex2++;

            // Crossover two better rules
            Rule crossoverRule=new Rule(betterRules[ruleIndex1]);
            int cutter=(int)(Math.random()*32);
            for (int j=cutter; j>=0; j--) {
              crossoverRule.set(j, betterRules[ruleIndex2].get(j));
            }
            newRules.set(x, y, crossoverRule);

            // Mutate
            newRules.set(x, y, newRules.get(x, y).mutate(MUTATION_PROBABILITY));
          }
        }
      }
      rules=newRules;
    }

    // End
    System.out.println("----------------------------------------------------");
    System.out.println("Best "+bestFitnesses);
    System.out.println("Best Fitness Success: "+bestFitnessSuccess+"%.");
    System.out.println("Best "+bestRules);
    System.out.println("Finished.");
  }

  /**
   * Finds all better rules of the specified with location one.
   * 
   * @param fitnesses the fitnesses grid (<code>null</code> not permitted).
   * @param rules the rules grid (<code>null</code> not permitted).
   * @param x the rule X coordinate.
   * @param y the rule Y coordinate.
   * @return the array of better rules (<code>null</code> possible).
   * @since 1.0
   */
  private Rule[] findBetterRules(Fitnesses fitnesses, Rules rules, int x, int y) {
    // Number of better rules
    int fitters=fitnesses.biggerNeighbours(x, y);
    if (fitters==0) return null; // No rule is better
    Rule[] betterRules=new Rule[fitters];

    // Add better rules to array
    int index=0;
    if (fitnesses.isAboveBigger(x, y)) {
      betterRules[index++]=rules.getAbove(x, y);
    }
    if (fitnesses.isRightBigger(x, y)) {
      betterRules[index++]=rules.getRight(x, y);
    }
    if (fitnesses.isBelowBigger(x, y)) {
      betterRules[index++]=rules.getBelow(x, y);
    }
    if (fitnesses.isLeftBigger(x, y)) {
      betterRules[index++]=rules.getLeft(x, y);
    }
    return betterRules;
  }
}
