/**
 * This cellular automata (CA) is a grid of cells, particularly its states. This
 * implementation is 2 dimensional.<br>
 * State can be <code>true</code>=1 or <code>false</code>=0.
 * 
 * @author TineL
 * @version 1.0, 2007-05-30
 * @since 1.0
 */
public class CA extends Abstract2DGrid {

  /** 2D array of cell states. */
  private boolean[][] ca;

  /**
   * Creates <code>CA</code> with the specified size. All cell states are set
   * to 0 (<code>false</code>).
   * 
   * @param width the CA width.
   * @param height the CA height.
   * @since 1.0
   */
  public CA(int width, int height) {
    super(width, height);
    this.ca=new boolean[height][width];
  }

  /**
   * Generates a CA with random cell states considering the ratio of 1s that
   * occur in the CA.
   * 
   * @param width the CA width.
   * @param height the CA height.
   * @param onesRatio the ratio of 1s that occur in the CA beside the 0s.
   * @return a random generated CA (<code>null</code> not possible).
   * @since 1.0
   */
  public static CA generateRandomCA(int width, int height, double onesRatio) {
    CA ca=new CA(width, height);
    for (int y=0; y<height; y++) {
      for (int x=0; x<width; x++) {
        ca.set(x, y, Math.random()<onesRatio?true:false);
      }
    }
    return ca;
  }

  /**
   * Generates a CA with random cell states considering the ratio of 1s that
   * occur in the CA only in the specified rectangle provided with boundaries.
   * Outside the rectangle cell states are 0s.
   * 
   * @param width the CA width.
   * @param height the CA height.
   * @param topEdge the index of the top-most rectangle boundary.
   * @param rightEdge the index of the right-most rectangle boundary.
   * @param bottomEdge the index of the bottom-most rectangle boundary.
   * @param leftEdge the index of the left-most rectangle boundary.
   * @param onesRatio the ratio of 1s that occur in the rectangle beside the 0s.
   * @return a random generated CA (<code>null</code> not possible).
   * @since 1.0
   */
  public static CA generateRandomCA(int width, int height, int topEdge,
    int rightEdge, int bottomEdge, int leftEdge, double onesRatio) {
    CA ca=new CA(width, height);
    for (int y=topEdge; y<=leftEdge; y++) {
      for (int x=bottomEdge; x<=rightEdge; x++) {
        ca.set(x, y, Math.random()<onesRatio?true:false);
      }
    }
    return ca;
  }

  /**
   * @param x the cell X coordinate.
   * @param y the cell Y coordinate.
   * @return the cell state.
   * @since 1.0
   */
  public boolean get(int x, int y) {
    return ca[y][x];
  }

  /**
   * Sets the state of the cell.
   * 
   * @param x the cell X coordinate.
   * @param y the cell Y coordinate.
   * @param state the cell new state.
   * @since 1.0
   */
  public void set(int x, int y, boolean state) {
    ca[y][x]=state;
  }

  /**
   * @param x the cell X coordinate.
   * @param y the cell Y coordinate.
   * @return the state of the above cell (or <code>false</code> if exceeds).
   * @since 1.0
   */
  public boolean getAbove(int x, int y) {
    int uy=getAboveIndex(y);
    if (uy==-1) return false;
    return get(x, uy);
  }

  /**
   * @param x the cell X coordinate.
   * @param y the cell Y coordinate.
   * @return the state of the right cell (or <code>false</code> if exceeds).
   * @since 1.0
   */
  public boolean getRight(int x, int y) {
    int rx=getRightIndex(x);
    if (rx==-1) return false;
    return get(rx, y);
  }

  /**
   * @param x the cell X coordinate.
   * @param y the cell Y coordinate.
   * @return the state of the below cell (or <code>false</code> if exceeds).
   * @since 1.0
   */
  public boolean getBelow(int x, int y) {
    int ly=getBelowIndex(y);
    if (ly==-1) return false;
    return get(x, ly);
  }

  /**
   * @param x the cell X coordinate.
   * @param y the cell Y coordinate.
   * @return the state of the left cell (or <code>false</code> if exceeds).
   * @since 1.0
   */
  public boolean getLeft(int x, int y) {
    int lx=getLeftIndex(x);
    if (lx==-1) return false;
    return get(lx, y);
  }

  /*
   * @since 1.0
   */
  public String toString() {
    String out="CA:";
    for (int y=0; y<getHeight(); y++) {
      out+="\n[";
      for (int x=0; x<getWidth(); x++) {
        out+=(get(x, y)==true?"1":"0");
      }
      out+="]";
    }
    return out;
  }
}
