/**
 * This grid is used as a superclass for some array-like classes. It is 2
 * dimensional.<br>
 * Spatially periodic boundary conditions are not applied, meaning that the grid
 * has boundary.
 * 
 * @author TineL
 * @version 1.0, 2007-05-30
 * @since 1.0
 */
public abstract class Abstract2DGrid {

  private int width=0;
  private int height=0;

  /**
   * Creates <code>Abstract2DGrid</code> with the specified size.
   * 
   * @param width the grid width.
   * @param height the grid height.
   * @since 1.0
   */
  public Abstract2DGrid(int width, int height) {
    this.width=width;
    this.height=height;
  }

  /**
   * @return the grid width.
   * @since 1.0
   */
  public int getWidth() {
    return this.width;
  }

  /**
   * @return the grid height.
   * @since 1.0
   */
  public int getHeight() {
    return this.height;
  }

  /**
   * @param index the index of the element in the grid.
   * @return the index of the above element (-1 if exceeds).
   * @throws IndexOutOfBoundsException if index is < 0 or >= grid height.
   * @since 1.0
   */
  public int getAboveIndex(int index) {
    if (index<0||index>=getHeight())
      throw new IndexOutOfBoundsException("Index must be [0, "+getHeight()+"]!");
    return index-1;
  }

  /**
   * @param index the index of the element in the grid.
   * @return the index of the right element (-1 if exceeds).
   * @throws IndexOutOfBoundsException if index is < 0 or >= grid width.
   * @since 1.0
   */
  public int getRightIndex(int index) {
    if (index<0||index>=getWidth())
      throw new IndexOutOfBoundsException("Index must be [0, "+getWidth()+"]!");
    return index+1>=getWidth()?-1:index+1;
  }

  /**
   * @param index the index of the element in the grid.
   * @return the index of the below element (-1 if exceeds).
   * @throws IndexOutOfBoundsException if index is < 0 or >= grid height.
   * @since 1.0
   */
  public int getBelowIndex(int index) {
    if (index<0||index>=getHeight())
      throw new IndexOutOfBoundsException("Index must be [0, "+getHeight()+"]!");
    return index+1>=getHeight()?-1:index+1;
  }

  /**
   * @param index the index of the element in the grid.
   * @return the index of the left element (-1 if exceeds).
   * @throws IndexOutOfBoundsException if index is < 0 or >= grid width.
   * @since 1.0
   */
  public int getLeftIndex(int index) {
    if (index<0||index>=getWidth())
      throw new IndexOutOfBoundsException("Index must be [0, "+getWidth()+"]!");
    return index-1;
  }
}
